Option EXPLICIT
BackLight 50
'
'  Written by Ian Thompson 8/2/2023 for Explore-100 as
'  an alternative Controller for the Solder Reflow Oven
'  minor timeout notification bug fixed 26/3/2023
'
'  using Geoff Graham's brilliant Project & GUI suite
'  (c.f. SC Micromite Plus Explore 100 Sep-Oct 2016)
'
'  Largely replicates Phil Prosser's Oven Project
'  (c.f. Silicon Chip Reflow Oven Project April-May 2020)
'
'  Provides better temperature overshoot control
'
'  Explore 100 loaded with MMBasic Ver 5.05.05
'
'  Saved variable definitions below
Dim integer i_t
Dim float i_g,i_o,i_p,i_i,i_d,delay1,i_st,i_pt,ramp5
Dim Float adv1,delay2,Xdeg,i_e,tcdec,scale
Font 4,2
'
'' Save VARiable settings to Default references?
'' ADD leading comment char next line to prevent the call
   reset_controls_to_defaults
'
' Constant and Variable definitions
Dim integer reflow, ans, changed, i, j, k, toggle, startT, savi
Dim integer start_offset, preheat, peak_temp,editsaved,redraw
Dim integer control_action, time_out, State, r
Dim integer tXY(2,6), flag, temp
Dim integer last_i, xptr, xptr_sav, dptr
Dim integer Tch = 77, P_ch = 95, P_int=92
Dim float refT, actT(800), tsum, eincr, decrement
Dim float integ, deriv, prop, ctl_act, last_T, T_err
Dim float Tset, last_sp, setP(2), esum, rrate, C(60)
Const c_Bwht=RGB(white), c_Bred= RGB(red)
Const c_wht=RGB(200,200,200), c_Bgreen=RGB(50,255,50)
Const true = 1, false = 0, x = 0, y = 1
Const t_set = 1, led = 2, t_sensor = 4, profile = 5, pid = 6
Const t_disp = 7, start = 8, editparams = 9
Const SENS_GAIN = 10, SENS_off = 11, GAIN_HDR = 12, OFF_HDR = 13
Const e_sens = 14, e_edit = 15, c_edit = 16, p_edit = 17
Const P_hdr =20, I_hdr = 21, D_hdr = 22, R_hdr = 23, E_hdr = 24
Const P_gain = 25, I_gain = 26, D_gain = 27
Const delay_hdr1=30, delay_hdr2=31, soak_hdr = 32, maxT_hdr = 33
Const del_tim1=35, soak_t = 37, peak_t = 38, f_ramp = 39
Const comp_gn=40, compdec=41, comp1=42,comp2=43,comp3=44
Const procset=46, procdeg=47, proc1=48,proc2=49,proc3=57
Const compset=59, compinc=60
Const a_bort=70, ref_hdr=71, t_refl=72, x_refl=73
Const t50= 80,t100=81,t150=82,t200=83,t250=84,s100=85,s200=86
Const s300=87,secs=88
Const header1=90,opthead1=91,opthead2=92,opthead3=93
Const sethead=94,meashead=95
Const sensor_hdr = "T_SENSOR SETTINGS"
Const PID_hdr = "EDIT P-I-D GAINS"
Const Profile_hdr = "SET REFLOW PROFILE"
Const reflow_hdr1 = "PROCESSING~REFLOW"
Const pwr2 = 100, pwr5 = 100          ' % of max. power

Data 5,5,5,3.6,3.4,3.6,4,4.4,4.7,5
Data 5.3,5.5,5.95,6.4,6.85,7.3,7.75,8.2,8.6,9
Data 9.45,9.85,10.25,10.65,11.05,11.45,11.8,12.3,12.75,13.2
Data 13.65,14.1,14.5,14.85,15.2,15.55,15.8,15.6,15.5,15.5
Data 15.4,15.9,16.5,17,17.6,18.2,18.8,19.4,20,20.9
Data 21.8,22.6,23.67
' load data
For i=1 To 53 : Read C(i) : Next i  ' sets look-up table

'  Set up Pins - 1xDOUT, 1xAIN, and 1xINTB input
SetPin Tch, AIN           ' analogue input channel
Pin(P_ch) = 1             ' preset this output high (SSR OFF)
SetPin P_ch, DOUT, OC     ' SSR-control\, OC output
SetPin P_int, INTB, pwrlight' interrupt input, to set CtrlVal(LED)

' Set up GUI controls (see Micromite Plus Manual)
GUI DELETE ALL
CLS
VAR RESTORE
'
' Start setting up GUI pages
'
GUI SETUP 1     ' Set up starting page 1 controls
Font 4,2
GUI CAPTION #50,"REFLOW OVEN CONTROLLER",30,10,,RGB(blue)
Font 5
GUI CAPTION #52,"SET-POINT:", 40,120,,c_Bwht
GUI CAPTION #53,"T_MEASURED:",40,235,,c_Bwht
Font 4,2
GUI SPINBOX t_set,305,90,350,80,RGB(RED),c_wht,1,20,200
CtrlVal(t_set)= i_t ' saved data
GUI DISPLAYBOX t_disp,385,210,193,80,c_Bred, c_wht
'GUI SWITCH #98,"ON|OFF",600,210,198,80,c_Bred,c_wht
'CtrlVal(#98)=false
GUI button start," START ~REFLOW", 560,345,235,130,c_Bred,c_wht
GUI button editparams,"EDITS", 5,345,200,100,c_Bred,c_wht

GUI SETUP 2
Font 4,2
GUI LED led, "HEAT",310,370,24,c_Bred

'GUI SETUP 3
'GUI LED #99, "TEST",280,260,20,RGB(RED)

GUI SETUP 4
Font 4,2
GUI CAPTION #54,SENSOR_HDR,10,10,,RGB(200, 20, 255)
Font 5
GUI CAPTION GAIN_HDR,"Sensor Gain:", 2,175,,c_Bwht
GUI CAPTION off_hdr,"Sensor Offset:",2,310,,c_Bwht
Font 4,2
GUI SPINBOX sens_gain,340,140,400,90,c_Bred,c_wht,0.005,5.8,6.5
CtrlVal(sens_gain)= i_g
GUI SPINBOX sens_off,340,280, 400,90,c_Bred,c_wht,0.5,-20,20
CtrlVal(sens_off)= i_o


GUI SETUP 5
Font 4,2
GUI CAPTION #55,PID_HDR,10,10,,RGB(cyan)
Font 5
GUI CAPTION P_HDR,"Proportional:", 5,145,,c_Bwht
GUI CAPTION I_hdr,"Integral:",5,275,,c_Bwht
GUI CAPTION D_hdr,"Derivative:",5,405,,c_Bwht
Font 4,2
GUI SPINBOX P_gain,320,120,320,80,c_Bred,c_wht,2,70,300
CtrlVal(P_gain)= i_p
GUI SPINBOX I_gain,320,250,320,80,c_Bred,c_wht,0.01,0,2
CtrlVal(I_gain)= i_i
GUI SPINBOX D_gain,320,380,320,80,c_Bred,c_wht,5,80,300
CtrlVal(D_gain)= i_d

GUI SETUP 6
Font 4,2
GUI CAPTION #56,PROFILE_HDR,2,20,,RGB(cyan)
Font 5
GUI CAPTION DELAY_HDR2,"DELAY1 (secs):",5,123,,c_Bwht
GUI CAPTION SOAK_hdr,  "Preheat T.`C:", 5,220,,c_Bwht
GUI CAPTION maxT_hdr,  "PEAK TEMP.`C:", 5,317,,c_Bwht
GUI CAPTION R_hdr,     "Ramp5 `C/sec:", 5,414,,c_Bwht
Font 4,2
GUI SPINBOX del_tim1,340,108,320,80,c_Bred,c_wht,1,0,40
CtrlVal(del_tim1) = delay1
GUI SPINBOX soak_t, 340,203,320,80,c_Bred,c_wht,5,100,170
CtrlVal(soak_t)= i_st
GUI SPINBOX peak_t, 340,298,320,80,c_Bred,c_wht,1,200,250
CtrlVal(peak_t)= i_pt
GUI SPINBOX f_ramp, 340,393,320,80,c_Bred,c_wht,0.1,0.8,1.3
CtrlVal(f_ramp)= ramp5

GUI SETUP 7
Font 4,2
GUI CAPTION ref_hdr,REFLOW_HDR1,90,47,,RGB(200, 20, 255)
GUI button a_bort,"ABORT", 530,330,190,70,c_Bred,c_wht
Font 5
GUI DISPLAYBOX x_refl,490,274,122,46,c_Bred, c_wht
GUI DISPLAYBOX t_refl,625,274,122,46,c_Bred, c_wht
Font 3
GUI CAPTION t50, " 50`", 15,408,,c_Bwht
GUI CAPTION t100,"100`", 15,308,,c_Bwht
GUI CAPTION t150,"150`", 15,208,,c_Bwht
GUI CAPTION t200,"200`", 15,108,,c_Bwht
GUI CAPTION t250,"250`", 15,  8,,c_Bwht
GUI CAPTION s100,"100", 200,460,"CTU",c_Bwht
GUI CAPTION s200,"200", 400,460,"CTU",c_Bwht
GUI CAPTION s300,"300", 600,460,"CTU",c_Bwht
GUI CAPTION secs,"seconds",650,430,,c_Bwht

GUI SETUP 8
Font 5
GUI FRAME opthead1,"EDIT OPTIONS 1",30,40,460,420,c_Bgreen
GUI button t_sensor,"Temp. Sensor", 70,80,380,100,c_Bred,c_wht
GUI button profile,"Temp PROFILE", 70,205,380,100,c_Bred,c_wht
GUI button pid,"PID settings", 70,330,380,100,c_Bred,c_wht
GUI button p_edit,"> Proc", 550,205,200,100,c_Bred,c_wht
GUI button c_edit,"> Comp", 550,330,200,100,c_Bred,c_wht

GUI SETUP 9
Font 4,2
GUI CAPTION opthead2,"Process Parameters",2,20,,RGB(cyan)
Font 5
GUI CAPTION DELAY_HDR1,"ADV1 (secs):",5,145,,c_Bwht
GUI CAPTION procset,"Delay2 (secs) :",5,275,,c_Bwht
GUI CAPTION procdeg,"Xdeg (cool)`C :",5,405,,c_Bwht
Font 4,2
GUI SPINBOX proc1,360,120,320,80,c_Bred,c_wht,1,0,50
CtrlVal(proc1) = adv1
GUI SPINBOX proc2,360,250,320,80,c_Bred,c_wht,1,20,60
CtrlVal(proc2) = delay2
GUI SPINBOX proc3,360,380,320,80,c_Bred,c_wht,1,10,50
CtrlVal(proc3) = Xdeg

GUI SETUP 10
Font 4.2
GUI button e_edit,"EXIT", 620,5,180,80,c_Bred,c_wht


GUI SETUP 11
Font 4,2
GUI CAPTION opthead3,"Compensation Data",2,20,,RGB(cyan)
Font 5
GUI CAPTION compinc,"Energy base:",5,145,,c_Bwht
GUI CAPTION compdec,"Base decay :",5,275,,c_Bwht
GUI CAPTION compset,"Dec scaling%:",5,405,,c_Bwht
Font 4,2
GUI SPINBOX comp1,360,120,350,80,c_Bred,c_wht,0.5,10,80
CtrlVal(comp1)= i_e
GUI SPINBOX comp2,360,250,350,80,c_Bred,c_wht,0.02,0.1,0.44
CtrlVal(comp2) = tcdec
GUI SPINBOX comp3,360,380,350,80,c_Bred,c_wht,2,50,150
CtrlVal(comp3) = scale
'
'  Set for Pages 1 & 2, set GUI interrupt, start 50ms ticks
Font 4,2
Page 1,2
GUI INTERRUPT Pg1DWN
Timer = 0
SetTick 500,tick,1  ' start 1/2 sec timer ticks


'****************** MAIN outer PROGRAM LOOP starts here *************

Do
' pre-setups for STATIC loop
  toggle = false
  last_i = 0: dptr = 0 : tsum=0 : integ = 0 : esum = 0
  control_action = true : time_out = 0 : reflow = false
  changed = false : editsaved = true : eincr = i_e
  tsum = get_volts() * i_g + i_o ' conv. volts*32 to temperature
  CtrlVal(t_set) = i_t           ' restore to start temp
  CtrlVal(t_disp)= Str$(tsum,3,1) ' display sensed temperature
  For i = 0 To 4
    actT(i) = tsum       'to initialise circular store
  Next i
  flag = false : decrement = 0
'****************** 1st page loop, controlling SET_POINT temperature
  Do
    If Not editsaved Then save_parameters
    tsum = 0.8*tsum + 0.2*(get_volts() * i_g + i_o) ' filter temp
    CtrlVal(t_disp)= Str$(tsum,3,1) ' display sensed temperature
    Pause 50
    If (control_action And (time_out > 3600)) Then
      control_action = false
      Pin(P_ch) = 1           ' turn off power right away
      r = MsgBox("TIMED OUT!~~Continue?", "YES", "NO")
      If r = 1 Then control_action = true : time_out = 0
    EndIf

    If (flag And control_action) Then
      flag = false
      tsum = get_volts() * i_g + i_o ' conv. volts*32 to temperature
      actT(dptr) = tsum
      Tset = CtrlVal(t_set)
      CtrlVal(t_disp)= Str$(tsum,3,1) ' display sensed temperature
      dptr = (dptr + 1) Mod 5         ' pointer cycles through 0,1,2
                                      ' to provide 2 second lookback
      last_T  = actT(dptr)            ' temperature 2 second ago
      deriv = (last_T - tsum)*i_d     ' deriv.(don't use target rate)
      T_err =  Tset - tsum
      prop  = T_err * i_p
      integ = integ + T_err * i_i
      If integ>30 Then integ = 30
      If integ<-30 Then integ = -30
      ctl_act = prop + integ + deriv
      If ctl_act > 10 Then
        ctl_act = ctl_act - esum
      Else
        esum = 0
      EndIf


'   Take Set Temp Control Action below
      If ctl_act > 50 Then
        Pin(P_ch) = 0       ' turn SSR on for 500 ms cyle time
        esum = esum + eincr
        decrement=getdecfig()     ' update decrement constant
        eincr = eincr - tcdec     ' slow energy incrementing rate
        If eincr < 4 Then eincr = 4
      ElseIf ctl_act => 2 Then    ' still <= 50
        Pin(P_ch)=1 'make sure pin is OFF\, so Pulse works properly
        temp = Cint(ctl_act*9)    ' = 450ms pulse if ctl_act = 50
        Pulse P_ch, temp
        Print "pulse=";Str$(temp,3);"ms"
        Print
'        esum = esum +  eincr * temp/1800
        decrement=getdecfig()
'        eincr = eincr - tcdec* temp/1800
        If eincr < 4 Then eincr = 4
      Else
        Pin(P_ch) = 1 ' no pulse required, so
        eincr = i_e   ' reset to energy base term
        esum = esum - decrement ' decay Esum
        If esum < 0 Then esum = 0  ' prevents a negative
      EndIf
      If (time_out Mod 2) = 0 Then
        Print "time= ";Str$(Cint(time_out/2),0);Str$(prop,10,0);
        Print Str$(integ,8,0); Str$(deriv,8,0);
        Print Str$(ctl_act,15,0);Str$(esum,15,0);
        Print Str$(decrement,5,2)
      EndIf
'      Print "Timer = ", Timer    ' for timing diagnostics
      Print
    EndIf
  Loop Until reflow

' ********** Reflow Loop set-up
  draw_reflow_graph_axes
  last_i = 0: xptr = 0 : tsum=0 : flag = false
  Do : Loop Until flag
  tsum = get_volts() * i_g + i_o ' conv. volts*32 to temperature
  actT(0) = tsum
  calc_ref_t    ' uses initially measured temperature
  draw_reflow_graph_lines
  redraw = false : decrement = 0
  control_action = true : toggle = false
  integ = 0 : esum = 0 : State = 1 : rrate = 0
  flag = false      ' start with a fresh tick

' *********  Start of REFLOW LOOP  **********
  Do
  If flag Then
    flag = false
    tsum = get_volts() * i_g + i_o ' conv. volts*32 to temperature
    actT(xptr) = tsum
'     Display temperature and time on reflow screen DISPLAYBOX's
    CtrlVal(t_refl) = Str$(tsum,3,0) + Chr$(96)
    CtrlVal(x_refl) = Str$(xptr/2,3,0) + "s"

    If actT(xptr) >= i_pt-8 Then   ' IF peak temperature approached
      control_action = false       ' stop controlling
      Pin(P_ch) = 1                ' turn the power OFF\
    EndIf

    If actT(xptr) >= i_pt Then   ' IF peak temperature reached
      GUI BEEP 50
    EndIf

    Select Case xptr
      Case IS > 4
        last_T = actT(xptr-4)     ' temperature 2 seconds ago
      Case 2, 3
        last_T = actT(xptr-2)     ' temp 1 second ago
      Case Else
        last_T = actT(xptr)       ' no initial derivative
    End Select

Print "State= ";Str$(State,0);",   Time= "; Str$(xptr/2,0,1);" secs";
Print "Temp= " ;Str$(tsum,0,1)

'   Step through States - act on results
    toggle = Not toggle
    If control_action Then
      Select Case State
        Case 1                        ' initially high power
          Pin(P_ch) = 0
          rrate = rrate*0.8 + (tsum-last_T)*0.2
          If rrate >= 1.2 Then State = 2 ' ramp achieved
          If (i_st-tsum) < Xdeg Then State = 3  ' within Xdeg o/ride
        Case 2                        ' set ramp sustain power
          If pwr2 > 96 Then
            Pin(P_ch)=0
          Else If pwr2 > 93 Then
            If toggle Then       ' alternate between PW=500 & 450ms
              Pin(P_ch) =0
            Else
              Pin(P_ch) =1
              Pulse P_ch, 450
            EndIf
          Else
            Pin(P_ch) = 1
            Pulse P_ch, pwr2*500/100  ' sets 450ms if pwr2=90%
          EndIf
          If (i_st-tsum) < Xdeg Then State = 3  ' switch OFF
        Case 3                        ' power off until soak temp
          Pin(P_ch) = 1
          If xptr >= (tXY(x,2) + delay2*2) Then State = 4
          integ = 0
        Case 4                        ' PID after settle time
          deriv = (last_T - tsum) * i_d ' derivative term
          T_err = i_st - tsum
          prop  = T_err * i_p
          integ = integ + T_err * i_i
          If integ>30 Then integ = 30
          If integ<-30 Then integ = -30
          ctl_act = prop + integ + deriv
          Print Str$(prop,5,0);Str$(integ,5,0);Str$(deriv,6,0);
          Print "   ="; Str$(ctl_act,0,0);
          If ctl_act > 10 Then
            ctl_act = ctl_act - esum
          Else
            esum = 0
          EndIf
          Print " Esum="; esum
          If ctl_act > 50 Then
            Pin(P_ch) = 0       ' turn SSR on for 500 ms cyle time
            esum = esum + eincr
            decrement=getdecfig()     ' update decrement constant
            eincr = eincr - tcdec
            If eincr < 4 Then eincr = 4
          ElseIf ctl_act => 2 Then
            Pin(P_ch)=1 'set pin OFF\, so Pulse works properly
            temp = Cint(ctl_act*9)  ' = 450ms pulse if ctl_act = 150
            Pulse P_ch, temp
            Print "pulse=";Str$(temp,3);"ms"
            Print
            esum = esum +  eincr * temp/2500
            decrement=getdecfig()
            eincr = eincr - tcdec* temp/2500
            If eincr < 4 Then eincr = 4
          Else
            Pin(P_ch) = 1 ' no pulse required, so
            eincr = i_e   ' reset to energy increment base term
            esum = esum - decrement
            If esum < 0 Then esum = 0  ' prevents a negative
          EndIf
          If xptr > (tXY(x,3)-adv1*2) Then State = 5
        Case 5                    ' high power to reflow
          If pwr5 > 95 Then
            Pin(P_ch)=0
          Else If pwr5 > 90 Then
            If toggle Then
              Pin(P_ch) =0
            Else
              Pin(P_ch) =1
              Pulse P_ch, 450
            EndIf
          Else
            Pin(P_ch) = 1
            Pulse P_ch, pwr5*500/100
          EndIf
        End Select
    EndIf
    draw_temp
    If xptr < 799 Then xptr = xptr + 1  ' prevents overflow
    If xptr = 799 Then Pin(P_ch)=1 : control_action = false
'    Print "Timer = ",Timer
    Print
  EndIf   ' executes if flag is set (on tick), loops otherwise
  Loop Until reflow=false
' ************** end of REFLOW LOOP
'
Loop  ' back to top of outer loop (back to STATIC control)
'
'*************** END OF OUTER PROGRAM LOOP


' ************ SUBROUTINES & FUNCTION DEFINITIONS below *********

Sub tick           ' on 0.5 second tick interrupt
  flag = true
  If control_action Then time_out=time_out+1 ' counting
'  Print "ticked,  ", Timer  ' for diagnostics
End Sub

Sub pwrlight       ' Interrupt sets the LED control value on Page 2
  CtrlVal(led) = Not Pin(P_int)
End Sub

Function get_volts() As FLOAT ' sums 32 voltage readings
  Local INTEGER k
  get_volts = 0
  For k = 1 To 32
    get_volts = get_volts + Pin(tCH)
  Next k
End Function

Function intY(disp As FLOAT) As INTEGER
  intY = Cint(479-(disp-20)*2)
End Function

Function getdecfig() As Float
  Local INTEGER ptr
  If Esum > 5200 Then Esum = 5200
  ptr = Cint(Esum/100 + 1)    ' max index = 53
  getdecfig = C(ptr)
End Function


Sub draw_temp
  j = intY(tsum)   ' convert to screen coordinates
  If xptr < 798 Then
    Line xptr,j-1,xptr,j+1,3,c_Bgreen ' draw pixels on screen
  Else
    Line 798, j, 799, j,3,c_Bgreen
  EndIf
  If redraw Then draw_reflow_graph_lines
End Sub

Sub draw_reflow_graph_axes    ' draw reflow page axes
  Line 0,0,0,480,3,c_Bwht
  Line 0,477,800,477,3,c_Bwht

  Line 0,420,10,420,3,c_Bwht ' 50
  Line 0,320,10,320,3,c_Bwht '100
  Line 0,220,10,220,3,c_Bwht '150
  Line 0,120,10,120,3,c_Bwht '200
  Line 0, 20,10, 20,3,c_Bwht '250

  Line 200,480,200,470,3,c_Bwht
  Line 400,480,400,470,3,c_Bwht
  Line 600,480,600,470,3,c_Bwht
End Sub

' Subroutine below determines reflow Target temperatures
' Saves line intercept points, to allow display refresh
Sub calc_ref_t ' parameters are 0.5 sec and 0.5 deg C /pixel
  startT = tsum
'  preheat=i_st-0.3
' Initial adjustable flat delay section
  tXY(x,0) = 0             ' start of trace at startT temp.
  tXY(y,0) = intY(startT)
' End of flat trace
  tXY(x,1) = delay1*2-1    ' convert to 1/2 secs intervals
  tXY(y,1) = tXY(y,0) ' end of flat, beginning of ramp

' 1st ramp, up to preheat soak temperature
  tXY(x,2) = tXY(x,1) + Cint((i_st-startT)/0.6) +1
  tXY(y,2) = intY(i_st)

' preheat flat section at i_st temperature
  tXY(x,3) = tXY(x,2) + 240
  tXY(y,3) = tXY(y,2)

' now ramp to peak temp i_pt
  tXY(x,4) = tXY(x,3) + Cint((i_pt-i_st)/(ramp5/2.0)) +1
  tXY(y,4) = intY(i_pt)

' hold peak temp briefly
  tXY(x,5) = tXY(x,4) + 8
  tXY(y,5) = tXY(y,4)

'  Shows suggested cool-down (needs oven door opened)
'  Controller will beep once peak temp is reached
'  and turn power off ready for cooling
  tXY(x,6) = 799
  tXY(y,6) = intY(i_pt - (799-tXY(x,5))*0.4) 'cool down target
End Sub

Sub draw_reflow_graph_lines ' redraws the graphed line, if needed
  Local INTEGER j
  Line 85,220,800,220,1,RGB(150,150,150)
  Line 5,70,800,70,1,RGB(150,150,150)
  For i = 1 To 6
    j = i-1
    If ((i=1) Or (i=3) Or (i=5)) Then
      Line tXY(x,j),tXY(y,j),  tXY(x,i),tXY(y,i)  ,3,c_Bred
    Else
      Line tXY(x,j),tXY(y,j)+1,tXY(x,i),tXY(y,i)+1, ,c_Bred
      Line tXY(x,j),tXY(y,j)  ,tXY(x,i),tXY(y,i)  , ,c_Bred
      Line tXY(x,j),tXY(y,j)-1,tXY(x,i),tXY(y,i)-1, ,c_Bred
    EndIf
  Next i
  For i = 0 To xptr   ' draw saved measured temps
    j = intY(actT(i))
    Line i,j-1,i,j+1,3,c_Bgreen
  Next i
  redraw=false    ' just redrawn
End Sub

Sub save_parameters ' Requests operator input to save variables
  i_t=CtrlVal(t_set)
  i_g=CtrlVal(sens_gain):i_o=CtrlVal(sens_off)
  i_p=CtrlVal(P_gain):i_i=CtrlVal(I_gain):i_d=CtrlVal(D_gain)
  ramp5=CtrlVal(f_ramp)
  delay1=CtrlVal(del_tim1):i_st=CtrlVal(soak_t):i_pt=CtrlVal(peak_t)
  adv1=CtrlVal(proc1):delay2=CtrlVal(proc2):Xdeg=CtrlVal(proc3)
  i_e=CtrlVal(comp1):tcdec=CtrlVal(comp2):scale=CtrlVal(comp3)
  If MsgBox("SAVE changes?","YES","NO") = 1 Then
     VAR SAVE i_g,i_o   ' don't save i_t, to start with default
     VAR SAVE i_p,i_i,i_d
     VAR SAVE delay1,i_st,i_pt,ramp5
     VAR SAVE adv1,delay2,Xdeg
     VAR SAVE i_e,tcdec,scale
  EndIf
  changed = false
  editsaved= true
End Sub

' ******************* BELOW are the GUI INTERRUPT routines
Sub Pg1DWN
  Select Case Touch(REF)
    Case editparams
      Page 8,10
      GUI INTERRUPT Pg8DWN
    Case start
      If MsgBox("Start REFLOW?","YES","NO") = 2  Then Exit Sub
      redraw=true
      Page 7,2
      GUI INTERRUPT Pg7DWN
      reflow = true
    Case t_set
      integ = 0   ' reset integral term
'    Case 98                              ' used for diagnostics
'      running = CtrlVal(#98)
'      If Not running Then Pin(P_ch) = 1 ' about to stop
    Case led  ' get back to STATIC control page from timeout
      If Not control_action Then CPU RESTART
  End Select
End Sub

Sub Pg4DWN
  Select Case Touch(REF)
    Case sens_gain, sens_off
      changed = true
    Case e_edit
      Page 8,10
      GUI INTERRUPT Pg8DWN
  End Select
End Sub

Sub Pg5DWN
  Select Case Touch(REF)
    Case P_gain, I_gain, D_gain
      changed = true
    Case e_edit
      Page 8,10
      GUI INTERRUPT Pg8DWN
  End Select
End Sub

Sub Pg6DWN
  Select Case Touch(REF)
    Case del_tim1, soak_t, peak_t, f_ramp
      changed = true
    Case e_edit
      Page 8,10
      GUI INTERRUPT Pg8DWN
  End Select
End Sub

Sub Pg7DWN
  Select Case Touch(REF)
    Case a_bort
      If MsgBox("Abort REFLOW action?","YES","NO") = 2  Then
        redraw=true
        Exit Sub
      EndIf
      reflow = false
      CtrlVal(led) = 0
      CLS
      GUI SHOW ALL
      Page 1,2
      GUI INTERRUPT Pg1DWN
  End Select
End Sub

Sub Pg8DWN
  Select Case Touch(REF)
    Case p_edit
      Page 9,10
      GUI INTERRUPT Pg9DWN
    Case c_edit
      Page 11,10
      GUI INTERRUPT Pg11DWN
    Case t_sensor
      Page 4,10
      GUI INTERRUPT Pg4DWN
    Case profile
      Page 6,10
      GUI INTERRUPT Pg6DWN
    Case pid
      Page 5,10
      GUI INTERRUPT Pg5DWN
    Case e_edit
      Page 1,2
      GUI INTERRUPT Pg1DWN
      If changed Then editsaved = false
  End Select
End Sub

Sub Pg9DWN
  Select Case Touch(REF)
    Case proc1, proc2, proc3  ', proc4
      changed = true
    Case e_edit
      Page 8,10
      GUI INTERRUPT Pg8DWN
  End Select
End Sub

Sub Pg11DWN
  Select Case Touch(REF)
    Case comp1, comp2, comp3  ', comp4
      changed = true
    Case e_edit
      Page 8,10
      GUI INTERRUPT Pg8DWN
  End Select
End Sub

Sub reset_controls_to_defaults
  Local M$ = "RESET Controls~to DEFAULTS !~~Are you sure?"
  Local integer r
  If MsgBox(M$,"YES","NO")=1 Then
    VAR CLEAR
    i_t=20:i_g=6.08:i_o=1                 ' set_T + Page 4, T_sensor
    i_p=140:i_i=1:i_d=125                 ' Page 5, PID settings
    delay1=31:i_st=150:i_pt=225:ramp5=1.0 ' Page 6, Profile
    adv1=40:delay2=28:Xdeg=29             ' Page 9, Processing
    i_e=57.5:tcdec=0.18:scale=124         ' Page 11, Compensation
'
    VAR SAVE i_g,i_o
    VAR SAVE i_p,i_i,i_d
    VAR SAVE delay1,i_st,i_pt,ramp5   ' save the VARiables
    VAR SAVE adv1,delay2,Xdeg
    VAR SAVE i_e,tcdec,scale
  Else
    r=MsgBox("You could comment~out this~subroutine call!","Ok")
  EndIf
End Sub                                                                                                 